<?php
/**
 * @Copyright
 * @package     EFSEO - Easy Frontend SEO for Joomal! 3.x
 * @author      Viktor Vogel <admin@kubik-rubik.de>
 * @version     3.4.0 - 2018-03-31
 * @link        https://joomla-extensions.kubik-rubik.de/efseo-easy-frontend-seo
 *
 * @license     GNU/GPL
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
defined('_JEXEC') || die('Restricted access');

use Joomla\Registry\Registry;

class EasyFrontendSeoModelEntry extends JModelLegacy
{
    protected $data = null;
    protected $id = null;
    protected $input;
    protected $error;

    public function __construct()
    {
        parent::__construct();

        $this->input = JFactory::getApplication()->input;

        $array = $this->input->get('id', 0, 'ARRAY');
        $this->setId((int) $array[0]);
    }

    /**
     * Gets the lengths of characters for title and description
     *
     * @return array
     */
    public function getCharactersLength()
    {
        $charactersLength = array('title' => 65, 'description' => 300);
        $plugin = JPluginHelper::getPlugin('system', 'easyfrontendseo');

        if (!empty($plugin)) {
            $pluginParams = new Registry();
            $pluginParams->loadString($plugin->params);

            $charactersLength = array('title' => $pluginParams->get('characters_title'), 'description' => $pluginParams->get('characters_description'));
        }

        return $charactersLength;
    }

    /**
     * Return the ID of the entry
     *
     * @return null
     */
    public function getId()
    {
        return $this->id;
    }

    /**
     * Sets the ID of the entry
     *
     * @param $id
     */
    public function setId($id)
    {
        $this->id = $id;
        $this->data = null;
    }

    /**
     * Returns error message - used by the controller
     *
     * @param null $i
     * @param bool $toString
     *
     * @return mixed
     */
    public function getError($i = null, $toString = true)
    {
        return $this->error;
    }

    /**
     * Loads the data from the database
     *
     * @return bool|JTable|mixed|null|stdClass
     * @throws Exception
     */
    public function getData()
    {
        if ($this->state->get('task') != 'add') {
            $this->data = $this->getInputSession();

            if (empty($this->data)) {
                $query = "SELECT * FROM " . $this->_db->quoteName('#__plg_easyfrontendseo') . " WHERE " . $this->_db->quoteName('id') . " = " . $this->_db->quote($this->id);
                $this->_db->setQuery($query);
                $this->data = $this->_db->loadObject();

                if (!empty($this->data)) {
                    return $this->data;
                }
            }
        }

        $this->data = $this->getTable('entry', 'EasyFrontendSeoTable');
        $this->data->id = 0;

        return $this->data;
    }

    /**
     * Restores the inpute data from the session
     *
     * @return bool|stdClass
     */
    private function getInputSession()
    {
        $session = JFactory::getSession();
        $input = $session->get('efseo_data');

        if (!empty($input)) {
            $data = new stdClass();

            $data->id = $input->getInt('id');
            $data->url = trim($input->get('url', '', 'RAW'));
            $data->title = trim($input->get('title', '', 'RAW'));
            $data->description = trim($input->get('description', '', 'RAW'));
            $data->keywords = trim($input->get('keywords', '', 'RAW'));
            $data->generator = trim($input->get('generator', '', 'RAW'));
            $data->robots = trim($input->get('robots', '', 'RAW'));

            $session->clear('efseo_data');

            return $data;
        }

        return false;
    }

    /**
     * Saves the entry into the database
     *
     * @return bool
     * @throws Exception
     */
    public function store()
    {
        $row = $this->getTable('entry', 'EasyFrontendSeoTable');
        $data = array();

        // Get entered data
        $data['id'] = $this->input->get('id', '', 'INT');
        $data['url'] = trim($this->input->get('url', '', 'STRING'));
        $data['title'] = trim($this->input->get('title', '', 'STRING'));
        $data['description'] = trim(stripslashes(preg_replace('@\s+(\r\n|\r|\n)@', ' ', $this->input->get('description', '', 'STRING'))));
        $data['keywords'] = trim($this->input->get('keywords', '', 'STRING'));
        $data['generator'] = trim($this->input->get('generator', '', 'STRING'));
        $data['robots'] = trim($this->input->get('robots', '', 'STRING'));

        // Do not save same URLs more than once
        if ($this->checkEntry($data['url']) == false) {
            $this->error = 'duplicate';

            return false;
        }

        if (!$row->save($data)) {
            $this->error = 'database';

            return false;
        }

        return true;
    }

    /**
     * Checks whether an URL already exists
     *
     * @param $url
     *
     * @return bool
     */
    private function checkEntry($url)
    {
        $db = JFactory::getDbo();

        $query = "SELECT * FROM " . $db->quoteName('#__plg_easyfrontendseo') . " WHERE " . $db->quoteName('url') . " = " . $db->quote($url);
        $this->_db->setQuery($query);
        $row = $this->_db->loadAssoc();

        if (empty($row)) {
            return true;
        }

        if ($row['id'] == $this->id) {
            return true;
        }

        return false;
    }

    /**
     * Updates the entries from the database in a batch process
     *
     * @return bool
     * @throws Exception
     */
    public function batch()
    {
        $ids = $this->input->get('id', 0, 'ARRAY');
        $vars = $this->input->get('batch', array('generator' => '', 'robots' => '', 'scheme' => '', 'coretables' => ''), 'ARRAY');
        $row = $this->getTable('entry', 'EasyFrontendSeoTable');

        foreach ($ids as $id) {
            $row->reset();
            $row->load($id);

            if (!empty($vars['generator']['activated'])) {
                $row->generator = $vars['generator']['value'];
            }

            if (!empty($vars['robots']['activated'])) {
                $row->robots = $vars['robots']['value'];
            }

            if (!empty($vars['coretables']['content']) || !empty($vars['coretables']['menu'])) {
                $this->batchCoreTables($vars['coretables'], $row);
            }

            if (!empty($vars['scheme']['activated'])) {
                $this->batchScheme($vars['scheme']['value'], $row->url);
            }

            if (!$row->store()) {
                $this->error = 'database';

                return false;
            }
        }

        return true;
    }

    /**
     * Deletes the entries from the database
     *
     * @return bool
     * @throws Exception
     */
    public function delete()
    {
        $ids = $this->input->get('id', 0, 'ARRAY');
        $row = $this->getTable('entry', 'EasyFrontendSeoTable');

        foreach ($ids as $id) {
            if (!$row->delete($id)) {
                $this->error = 'database';

                return false;
            }
        }

        return true;
    }

    /**
     * Changes the state of the entries
     *
     * @param $state
     *
     * @return bool
     * @throws Exception
     */
    public function publish($state)
    {
        $id = $this->input->get('id', 0, 'ARRAY');
        $row = $this->getTable('entry', 'EasyFrontendSeoTable');

        if (!$row->publish($id, $state)) {
            $this->error = 'database';

            return false;
        }

        return true;
    }

    /**
     * Stores the input data into the session
     *
     * @param $input
     */
    public function storeInputSession($input)
    {
        $session = JFactory::getSession();
        $session->set('efseo_data', $input);

        return;
    }

    /**
     * Batch function to change the scheme (http / https) of an entry
     *
     * @param string $value
     * @param string $url
     */
    private function batchScheme($value, &$url)
    {
        // Process only absolute URLs
        if (preg_match('@^https?://@i', $url, $match)) {
            if ($value == 'HTTPS' && $match[0] == 'http://') {
                $url = str_replace($match[0], 'https://', $url);

                return;
            }

            if ($value == 'HTTP' && $match[0] == 'https://') {
                $url = str_replace($match[0], 'http://', $url);
            }
        }
    }

    /**
     * Batch function to write meta data directly to the core tables
     *
     * @param $coreTables
     * @param $itemData
     *
     * @throws Exception
     */
    private function batchCoreTables($coreTables, $itemData)
    {
        $requestVariables = array();
        $plugin = JPluginHelper::getPlugin('system', 'easyfrontendseo');
        $params = new Registry($plugin->params);
        $saveDataTableToken = $params->get('save_data_table_token');

        if (!empty($saveDataTableToken)) {
            $url = $itemData->url . (strpos($itemData->url, '?') !== false ? '&' : '?') . 'efseo_batch_token=' . rawurlencode($params->get('save_data_table_token'));

            if (strpos($url, JUri::root()) === false) {
                $url = JUri::root() . $url;
            }

            $request = JHttpFactory::getHttp()->get($url);

            if ($request->code == 200 && $request->headers['Content-Type'] == 'application/json') {
                $requestVariables = json_decode($request->body, true);
            }
        }

        if (!empty($requestVariables)) {
            if ($coreTables['content']) {
                $this->saveDataToTableContent($itemData, $requestVariables);
            }

            if ($coreTables['menu']) {
                $this->saveDataToTableMenu($itemData, $requestVariables);
            }
        }
    }

    /**
     * Saves data to the core content table
     *
     * @param object $itemData
     * @param array  $requestVariables
     */
    private function saveDataToTableContent($itemData, $requestVariables)
    {
        // Only execute if we are in the article view of the content component
        if ($requestVariables['option'] == 'com_content' && $requestVariables['view'] == 'article') {
            if (!empty($requestVariables['id'])) {
                $query = "UPDATE " . $this->_db->quoteName('#__content') . " SET " . $this->_db->quoteName('metakey') . " = " . $this->_db->quote($itemData->keywords) . ", " . $this->_db->quoteName('metadesc') . " = " . $this->_db->quote($itemData->description) . " WHERE " . $this->_db->quoteName('id') . " = " . $this->_db->quote((int) $requestVariables['id']);
                $this->_db->setQuery($query);
                $this->_db->execute();
            }
        }
    }

    /**
     * Saves data to the core menu table
     *
     * @param object $itemData
     * @param array  $requestVariables
     *
     * @throws Exception
     */
    private function saveDataToTableMenu($itemData, $requestVariables)
    {
        if (!empty($requestVariables['itemid'])) {
            $menu = JMenu::getInstance('site')->getItem((int) $requestVariables['itemid']);

            // Check whether menu entry for the specific item exists - e.g. do not overwrite data of blog entry
            foreach ($menu->query as $key => $value) {
                if (!empty($requestVariables[$key])) {
                    if ($value != $requestVariables[$key]) {
                        return;
                    }
                }
            }

            $menuParamsArray = JMenu::getInstance('site')->getParams($menu->id)->toArray();
            $menuParamsArray['page_title'] = $itemData->title;
            $menuParamsArray['menu-meta_description'] = $itemData->description;
            $menuParamsArray['menu-meta_keywords'] = $itemData->keywords;

            $menu_params = json_encode($menuParamsArray);

            $query = "UPDATE " . $this->_db->quoteName('#__menu') . " SET " . $this->_db->quoteName('params') . " = " . $this->_db->quote($menu_params) . " WHERE " . $this->_db->quoteName('id') . " = " . $this->_db->quote((int) $requestVariables['itemid']);
            $this->_db->setQuery($query);
            $this->_db->execute();
        }
    }
}
